@extends('layouts.adminfonts')

@section('title', 'Manage Fonts')
@section('page-title', 'Font Management')

@section('content')
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>

            <p class="text-muted mb-0">Upload and manage your custom fonts</p>
        </div>
        <a href="{{ route('admin.fonts.create') }}" class="btn btn-primary">
            <i class="fas fa-plus me-2"></i>
            Upload New Font
        </a>
    </div>

    <!-- Advanced Search Form -->
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="card-title mb-0">
                <i class="fas fa-search me-2"></i>
                Advanced Font Search
            </h5>
        </div>
        <div class="card-body">
            <form method="GET" action="{{ route('admin.fonts.index') }}" class="row g-3" id="searchForm">
                <div class="col-md-3">
                    <label for="search" class="form-label">Search Font Name</label>
                    <div class="input-group">
                        <span class="input-group-text">
                            <i class="fas fa-font"></i>
                        </span>
                        <input type="text" class="form-control" id="search" name="search"
                            value="{{ request('search') }}" placeholder="Enter font name...">
                    </div>
                </div>

                <div class="col-md-2">
                    <label for="display_name" class="form-label">Display Name</label>
                    <div class="input-group">
                        <span class="input-group-text">
                            <i class="fas fa-tag"></i>
                        </span>
                        <input type="text" class="form-control" id="display_name" name="display_name"
                            value="{{ request('display_name') }}" placeholder="Enter display name...">
                    </div>
                </div>

                <div class="col-md-1">
                    <label for="gender" class="form-label">Gender</label>
                    <select class="form-select" id="gender" name="gender">
                        <option value="">All Genders</option>
                        <option value="Male" {{ request('gender') == 'Male' ? 'selected' : '' }}>Male</option>
                        <option value="Female" {{ request('gender') == 'Female' ? 'selected' : '' }}>Female</option>
                        <option value="Unisex" {{ request('gender') == 'Unisex' ? 'selected' : '' }}>Unisex</option>
                    </select>
                </div>

                <div class="col-md-1">
                    <label for="size" class="form-label">Size</label>
                    <div class="input-group">
                        <input type="number" class="form-control" id="size" name="size"
                            value="{{ request('size') }}" placeholder="vw" min="5" max="50">
                    </div>
                </div>

                <div class="col-md-1">
                    <label for="uppercase_only" class="form-label">Uppercase</label>
                    <select class="form-select" id="uppercase_only" name="uppercase_only">
                        <option value="">All Fonts</option>
                        <option value="1" {{ request('uppercase_only') == '1' ? 'selected' : '' }}>Yes</option>
                        <option value="0" {{ request('uppercase_only') == '0' ? 'selected' : '' }}>No</option>
                    </select>
                </div>

                <div class="col-md-2">
                    <label for="tags" class="form-label">Tags</label>
                    <div class="input-group">
                        <span class="input-group-text">
                            <i class="fas fa-tags"></i>
                        </span>
                        <input type="text" class="form-control" id="tags" name="tags"
                            value="{{ request('tags') }}" placeholder="script, bold, vintage...">
                    </div>
                </div>

                <div class="col-md-1">
                    <label class="form-label">&nbsp;</label>
                    <div class="d-grid">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-search"></i>
                        </button>
                    </div>
                </div>
            </form>

            <!-- Active Filters Display -->
            @if (request()->hasAny(['search', 'display_name', 'gender', 'size', 'uppercase_only', 'tags']))
                <div class="mt-3">
                    <div class="d-flex align-items-center flex-wrap gap-2">
                        <span class="text-muted">Active filters:</span>
                        @if (request('search'))
                            <span class="badge bg-primary">
                                Font: {{ request('search') }}
                                <a href="{{ request()->fullUrlWithQuery(['search' => null]) }}" class="text-white ms-1">
                                    <i class="fas fa-times"></i>
                                </a>
                            </span>
                        @endif
                        @if (request('display_name'))
                            <span class="badge bg-info">
                                Display: {{ request('display_name') }}
                                <a href="{{ request()->fullUrlWithQuery(['display_name' => null]) }}"
                                    class="text-white ms-1">
                                    <i class="fas fa-times"></i>
                                </a>
                            </span>
                        @endif
                        @if (request('gender'))
                            <span class="badge bg-success">
                                Gender: {{ request('gender') }}
                                <a href="{{ request()->fullUrlWithQuery(['gender' => null]) }}" class="text-white ms-1">
                                    <i class="fas fa-times"></i>
                                </a>
                            </span>
                        @endif
                        @if (request('size'))
                            <span class="badge bg-warning text-dark">
                                Size: {{ request('size') }}vw
                                <a href="{{ request()->fullUrlWithQuery(['size' => null]) }}" class="text-dark ms-1">
                                    <i class="fas fa-times"></i>
                                </a>
                            </span>
                        @endif
                        @if (request('uppercase_only') !== null && request('uppercase_only') !== '')
                            <span class="badge bg-dark">
                                Uppercase: {{ request('uppercase_only') == '1' ? 'Yes' : 'No' }}
                                <a href="{{ request()->fullUrlWithQuery(['uppercase_only' => null]) }}"
                                    class="text-white ms-1">
                                    <i class="fas fa-times"></i>
                                </a>
                            </span>
                        @endif
                        @if (request('tags'))
                            <span class="badge bg-secondary">
                                Tags: {{ request('tags') }}
                                <a href="{{ request()->fullUrlWithQuery(['tags' => null]) }}" class="text-white ms-1">
                                    <i class="fas fa-times"></i>
                                </a>
                            </span>
                        @endif
                        <a href="{{ route('admin.fonts.index') }}" class="btn btn-sm btn-outline-secondary">
                            <i class="fas fa-times me-1"></i>
                            Clear All
                        </a>
                    </div>
                </div>
            @endif
        </div>
    </div>

    <div class="card">
        <div class="card-body">
            @if ($fonts->count() > 0)
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th width="120">Preview</th>
                                <th>Font Name</th>
                                <th>Display Name</th>
                                <th>Gender</th>
                                <th>Size</th>
                                <th>Uppercase Only</th>
                                <th>Status</th>
                                <th>Created</th>
                                <th width="200">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach ($fonts as $font)
                                <tr>
                                    <td>
                                        <div class="font-preview clickable-preview" style="{{ $font->style }}"
                                            data-font-style="{{ $font->style }}"
                                            data-font-name="{{ $font->font_name }}"
                                            data-display-name="{{ $font->display_name }}" title="Click to preview">
                                            Example
                                        </div>
                                    </td>
                                    <td>
                                        <code>{{ $font->font_name }}</code>
                                    </td>
                                    <td>{{ $font->display_name }}</td>
                                    <td>
                                        <span class="badge bg-info">{{ $font->gender }}</span>
                                    </td>
                                    <td>{{ $font->font_size }}vw</td>
                                    <td>
                                        @if ($font->uppercase_only)
                                            <span class="badge bg-success">
                                                <i class="fas fa-check me-1"></i>
                                                Yes
                                            </span>
                                        @else
                                            <span class="badge bg-secondary">
                                                <i class="fas fa-times me-1"></i>
                                                No
                                            </span>
                                        @endif
                                    </td>
                                    <td>
                                        <span
                                            class="status-badge {{ $font->is_active ? 'status-active' : 'status-inactive' }}">
                                            {{ $font->is_active ? 'Active' : 'Inactive' }}
                                        </span>
                                    </td>
                                    <td>{{ $font->created_at->format('M d, Y') }}</td>
                                    <td>
                                        <div class="action-buttons">
                                            <a href="{{ route('admin.fonts.show', $font) }}"
                                                class="btn btn-sm btn-outline-primary" title="View">
                                                <i class="fas fa-eye"></i>
                                            </a>
                                            <a href="{{ route('admin.fonts.edit', $font) }}"
                                                class="btn btn-sm btn-outline-warning" title="Edit">
                                                <i class="fas fa-edit"></i>
                                            </a>
                                            <form action="{{ route('admin.fonts.toggle', $font) }}" method="POST"
                                                class="d-inline">
                                                @csrf
                                                <button type="submit"
                                                    class="btn btn-sm {{ $font->is_active ? 'btn-outline-danger' : 'btn-outline-success' }}"
                                                    title="{{ $font->is_active ? 'Deactivate' : 'Activate' }}">
                                                    <i class="fas {{ $font->is_active ? 'fa-pause' : 'fa-play' }}"></i>
                                                </button>
                                            </form>
                                            <form action="{{ route('admin.fonts.destroy', $font) }}" method="POST"
                                                class="d-inline"
                                                onsubmit="return confirm('Are you sure you want to delete this font?')">
                                                @csrf
                                                @method('DELETE')
                                                <button type="submit" class="btn btn-sm btn-outline-danger"
                                                    title="Delete">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>

                <!-- Custom Pagination -->
                <div class="custom-pagination-wrapper">
                    <div class="pagination-info">
                        Showing {{ $fonts->firstItem() ?? 0 }} to {{ $fonts->lastItem() ?? 0 }} of {{ $fonts->total() }}
                        fonts
                    </div>

                    @if ($fonts->hasPages())
                        <div class="custom-pagination">
                            {{-- Previous Button --}}
                            @if ($fonts->onFirstPage())
                                <span class="pagination-btn disabled">
                                    <i class="fas fa-chevron-left"></i>
                                </span>
                            @else
                                <a href="{{ $fonts->previousPageUrl() }}" class="pagination-btn">
                                    <i class="fas fa-chevron-left"></i>
                                </a>
                            @endif

                            {{-- Page Numbers --}}
                            @php
                                $start = max($fonts->currentPage() - 2, 1);
                                $end = min($start + 4, $fonts->lastPage());
                            @endphp

                            @if ($start > 1)
                                <a href="{{ $fonts->url(1) }}" class="pagination-btn">1</a>
                                @if ($start > 2)
                                    <span class="pagination-dots">...</span>
                                @endif
                            @endif

                            @for ($i = $start; $i <= $end; $i++)
                                @if ($i == $fonts->currentPage())
                                    <span class="pagination-btn active">{{ $i }}</span>
                                @else
                                    <a href="{{ $fonts->url($i) }}" class="pagination-btn">{{ $i }}</a>
                                @endif
                            @endfor

                            @if ($end < $fonts->lastPage())
                                @if ($end < $fonts->lastPage() - 1)
                                    <span class="pagination-dots">...</span>
                                @endif
                                <a href="{{ $fonts->url($fonts->lastPage()) }}"
                                    class="pagination-btn">{{ $fonts->lastPage() }}</a>
                            @endif

                            {{-- Next Button --}}
                            @if ($fonts->hasMorePages())
                                <a href="{{ $fonts->nextPageUrl() }}" class="pagination-btn">
                                    <i class="fas fa-chevron-right"></i>
                                </a>
                            @else
                                <span class="pagination-btn disabled">
                                    <i class="fas fa-chevron-right"></i>
                                </span>
                            @endif
                        </div>
                    @endif
                </div>
            @else
                <div class="text-center py-5">
                    <i class="fas fa-font fa-4x text-muted mb-4"></i>
                    <h4 class="text-muted">No fonts uploaded yet</h4>
                    <p class="text-muted mb-4">Start building your font collection by uploading your first font.</p>
                    <a href="{{ route('admin.fonts.create') }}" class="btn btn-primary btn-lg">
                        <i class="fas fa-plus me-2"></i>
                        Upload Your First Font
                    </a>
                </div>
            @endif
        </div>
    </div>
@endsection

@section('scripts')
    <script>
        // Search form enhancements
        document.addEventListener('DOMContentLoaded', function() {
            // Debounced search for text inputs
            let searchTimeout;

            // Auto-submit form on select change
            document.getElementById('gender').addEventListener('change', function() {
                console.log('Gender changed:', this.value);
                document.getElementById('searchForm').submit();
            });

            document.getElementById('size').addEventListener('input', function() {
                console.log('Size input:', this.value);
                clearTimeout(searchTimeout);
                searchTimeout = setTimeout(() => {
                    console.log('Submitting form with size:', this.value);
                    document.getElementById('searchForm').submit();
                }, 500);
            });

            document.getElementById('uppercase_only').addEventListener('change', function() {
                console.log('Uppercase changed:', this.value);
                document.getElementById('searchForm').submit();
            });
            document.getElementById('search').addEventListener('input', function() {
                console.log('Search input:', this.value);
                clearTimeout(searchTimeout);
                searchTimeout = setTimeout(() => {
                    console.log('Submitting form with search:', this.value);
                    document.getElementById('searchForm').submit();
                }, 500);
            });

            document.getElementById('display_name').addEventListener('input', function() {
                console.log('Display name input:', this.value);
                clearTimeout(searchTimeout);
                searchTimeout = setTimeout(() => {
                    console.log('Submitting form with display_name:', this.value);
                    document.getElementById('searchForm').submit();
                }, 500);
            });

            document.getElementById('tags').addEventListener('input', function() {
                console.log('Tags input:', this.value);
                clearTimeout(searchTimeout);
                searchTimeout = setTimeout(() => {
                    console.log('Submitting form with tags:', this.value);
                    document.getElementById('searchForm').submit();
                }, 500);
            });

            // Clear search on Escape key
            document.addEventListener('keydown', function(e) {
                if (e.key === 'Escape') {
                    document.getElementById('search').value = '';
                    document.getElementById('display_name').value = '';
                    document.getElementById('gender').value = '';
                    document.getElementById('size').value = '';
                    document.getElementById('uppercase_only').value = '';
                    document.getElementById('tags').value = '';
                    document.getElementById('searchForm').submit();
                }
            });
        });

        // Auto-refresh preview on hover
        document.querySelectorAll('.font-preview').forEach(function(element) {
            element.addEventListener('mouseenter', function() {
                this.style.transform = 'scale(1.1)';
                this.style.transition = 'transform 0.3s ease';
            });

            element.addEventListener('mouseleave', function() {
                this.style.transform = 'scale(1)';
            });
        });

        // Simple modal test
        document.addEventListener('DOMContentLoaded', function() {
            console.log('DOM loaded, setting up font preview modals');

            // Add click listeners to all preview elements
            document.querySelectorAll('.clickable-preview').forEach(function(element) {
                element.addEventListener('click', function(e) {
                    e.preventDefault();
                    console.log('Preview clicked!');

                    // Get font data from data attributes
                    const fontStyle = this.getAttribute('data-font-style');
                    const fontName = this.getAttribute('data-font-name');
                    const displayName = this.getAttribute('data-display-name');

                    console.log('Font data:', {
                        fontStyle,
                        fontName,
                        displayName
                    });

                    if (fontStyle && fontName && displayName) {
                        // Call the global function
                        if (window.openFontPreview) {
                            window.openFontPreview(fontStyle, fontName, displayName);
                        } else {
                            console.error('openFontPreview function not found');
                            // Fallback: show test modal
                            const testModal = new bootstrap.Modal(document.getElementById(
                                'testModal'));
                            document.getElementById('testModalContent').innerHTML =
                                '<strong>Font Preview:</strong> ' + displayName + '<br>' +
                                '<strong>Font Name:</strong> ' + fontName + '<br>' +
                                '<strong>Style:</strong> ' + fontStyle;
                            testModal.show();
                        }
                    } else {
                        console.error('Missing font data attributes');
                    }
                });
            });
        });
    </script>
@endsection
